<?php
/**
 * Plugin Name: Dynamic City Google Map (from Title + CSV)
 * Description: Ajoute automatiquement une carte Google dans les articles en détectant une ville présente dans le titre (liste de villes depuis /uploads/villes.csv).
 * Version: 1.0.0
 * Author: (toi)
 */

if (!defined('ABSPATH')) exit;

class DCGM_Dynamic_City_Google_Map {
  const OPTION_KEY = 'dcgm_settings';
  const META_CITY  = '_dcgm_city_detected';     // ex: "Liège"
  const META_NORM  = '_dcgm_city_detected_norm';// ex: "liege"
  const CSV_REL_PATH = 'villes.csv';            // dans wp-content/uploads/

  public static function init() {
    add_action('admin_init', [__CLASS__, 'register_settings']);
    add_action('admin_menu', [__CLASS__, 'add_settings_page']);

    add_action('save_post', [__CLASS__, 'on_save_post'], 10, 2);

    add_filter('the_content', [__CLASS__, 'append_map_to_content'], 20);

    add_shortcode('city_map', [__CLASS__, 'shortcode_city_map']);
  }

  /* -------------------------
     Settings
  ------------------------- */
  public static function register_settings() {
    register_setting('dcgm', self::OPTION_KEY);

    $defaults = [
      'api_key' => '',
      'auto_append' => 1,     // ajoute automatiquement la carte
      'height' => 360,
      'country_hint' => 'Belgique',
      'cache_minutes' => 720, // 12h pour cache de la liste villes
    ];

    $opt = get_option(self::OPTION_KEY);
    if (!$opt) update_option(self::OPTION_KEY, $defaults);
  }

  public static function add_settings_page() {
    add_options_page(
      'Dynamic City Google Map',
      'City Google Map',
      'manage_options',
      'dcgm',
      [__CLASS__, 'render_settings_page']
    );
  }

  public static function render_settings_page() {
    if (!current_user_can('manage_options')) return;
    $opt = self::get_settings();
    ?>
    <div class="wrap">
      <h1>City Google Map</h1>
      <p>
        Le plugin lit la liste des villes ici :
        <code><?php echo esc_html(self::get_csv_path()); ?></code>
      </p>

      <form method="post" action="options.php">
        <?php settings_fields('dcgm'); ?>
        <table class="form-table" role="presentation">
          <tr>
            <th scope="row"><label for="dcgm_api_key">Google Maps API key</label></th>
            <td>
              <input id="dcgm_api_key" name="<?php echo esc_attr(self::OPTION_KEY); ?>[api_key]"
                     type="text" class="regular-text"
                     value="<?php echo esc_attr($opt['api_key']); ?>" />
              <p class="description">
                Utilise l’API <strong>Maps Embed API</strong> (iframe).
              </p>
            </td>
          </tr>

          <tr>
            <th scope="row">Ajout automatique</th>
            <td>
              <label>
                <input type="checkbox"
                       name="<?php echo esc_attr(self::OPTION_KEY); ?>[auto_append]"
                       value="1" <?php checked(!empty($opt['auto_append'])); ?> />
                Ajouter la carte automatiquement à la fin des articles
              </label>
              <p class="description">
                Sinon, tu peux placer <code>[city_map]</code> dans le contenu.
              </p>
            </td>
          </tr>

          <tr>
            <th scope="row"><label for="dcgm_height">Hauteur carte (px)</label></th>
            <td>
              <input id="dcgm_height" name="<?php echo esc_attr(self::OPTION_KEY); ?>[height]"
                     type="number" min="200" max="900"
                     value="<?php echo (int)$opt['height']; ?>" />
            </td>
          </tr>

          <tr>
            <th scope="row"><label for="dcgm_country">Indice pays</label></th>
            <td>
              <input id="dcgm_country" name="<?php echo esc_attr(self::OPTION_KEY); ?>[country_hint]"
                     type="text" class="regular-text"
                     value="<?php echo esc_attr($opt['country_hint']); ?>" />
              <p class="description">Ajouté à la requête: “Ville, Belgique” pour éviter les ambiguïtés.</p>
            </td>
          </tr>

          <tr>
            <th scope="row"><label for="dcgm_cache">Cache liste villes (minutes)</label></th>
            <td>
              <input id="dcgm_cache" name="<?php echo esc_attr(self::OPTION_KEY); ?>[cache_minutes]"
                     type="number" min="1" max="10080"
                     value="<?php echo (int)$opt['cache_minutes']; ?>" />
            </td>
          </tr>
        </table>

        <?php submit_button(); ?>
      </form>

      <hr />
      <h2>Test rapide</h2>
      <ol>
        <li>Assure-toi que le fichier <code>villes.csv</code> est bien dans <code>/uploads/</code></li>
        <li>Édite un article dont le titre contient une ville (ex: “Dépannage à Liège”)</li>
        <li>Affiche l’article : la carte doit apparaître (si “Ajout automatique” coché)</li>
      </ol>
    </div>
    <?php
  }

  private static function get_settings() {
    $opt = get_option(self::OPTION_KEY, []);
    $defaults = [
      'api_key' => '',
      'auto_append' => 1,
      'height' => 360,
      'country_hint' => 'Belgique',
      'cache_minutes' => 720,
    ];
    return array_merge($defaults, is_array($opt) ? $opt : []);
  }

  private static function get_csv_path() {
    $uploads = wp_upload_dir();
    return trailingslashit($uploads['basedir']) . self::CSV_REL_PATH;
  }

  /* -------------------------
     City list loading (CSV)
  ------------------------- */
  private static function load_city_map() {
    $opt = self::get_settings();
    $cache_key = 'dcgm_city_map_v1';
    $cached = get_transient($cache_key);
    if (is_array($cached) && !empty($cached)) return $cached;

    $path = self::get_csv_path();
    if (!file_exists($path)) return [];

    $cities = [];

    // lecture CSV (on accepte ; ou ,)
    $handle = fopen($path, 'r');
    if (!$handle) return [];

    // détecteur de delimiter simple
    $firstLine = fgets($handle);
    if ($firstLine === false) { fclose($handle); return []; }
    $delimiter = (substr_count($firstLine, ';') >= substr_count($firstLine, ',')) ? ';' : ',';
    rewind($handle);

    while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
      foreach ($row as $cell) {
        $cell = trim((string)$cell);
        if ($cell === '') continue;
        $norm = self::normalize($cell);
        if ($norm === '') continue;

        // on stocke la forme "affichable" (cellule originale)
        // (première occurrence gagnante)
        if (!isset($cities[$norm])) {
          $cities[$norm] = $cell;
        }
      }
    }
    fclose($handle);

    // trier par longueur desc pour matcher d'abord les noms composés
    uksort($cities, function($a, $b){
      return mb_strlen($b) <=> mb_strlen($a);
    });

    $ttl = max(1, (int)$opt['cache_minutes']) * MINUTE_IN_SECONDS;
    set_transient($cache_key, $cities, $ttl);

    return $cities;
  }

  /* -------------------------
     Normalization + detection
  ------------------------- */
  private static function normalize($s) {
    $s = (string)$s;
    $s = trim($s);
    if ($s === '') return '';

    // translit accents
    $t = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
    if ($t !== false) $s = $t;

    $s = mb_strtolower($s);
    // garder lettres/chiffres, convertir le reste en espaces
    $s = preg_replace('~[^a-z0-9]+~', ' ', $s);
    $s = trim(preg_replace('~\s+~', ' ', $s));
    return $s;
  }

  private static function detect_city_in_title($title) {
    $cities = self::load_city_map();
    if (empty($cities)) return [null, null];

    $normTitle = self::normalize($title);
    if ($normTitle === '') return [null, null];

    // "word boundary" simple via padding espaces
    $hay = ' ' . $normTitle . ' ';

    foreach ($cities as $norm => $display) {
      $needle = ' ' . $norm . ' ';
      if (strpos($hay, $needle) !== false) {
        return [$display, $norm];
      }
    }
    return [null, null];
  }

  /* -------------------------
     Save post: cache detected city in meta
  ------------------------- */
  public static function on_save_post($post_id, $post) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (wp_is_post_revision($post_id)) return;
    if (!$post || $post->post_type !== 'post') return;

    // si pas publié, on peut quand même calculer
    [$city, $norm] = self::detect_city_in_title(get_the_title($post_id));

    if ($city) {
      update_post_meta($post_id, self::META_CITY, $city);
      update_post_meta($post_id, self::META_NORM, $norm);
    } else {
      delete_post_meta($post_id, self::META_CITY);
      delete_post_meta($post_id, self::META_NORM);
    }
  }

  /* -------------------------
     Build iframe HTML
  ------------------------- */
  private static function build_map_html($city) {
    $opt = self::get_settings();
    $key = trim((string)$opt['api_key']);
    if ($key === '') return ''; // sans clé => pas de map

    $height = max(200, (int)$opt['height']);
    $country = trim((string)$opt['country_hint']);
    $q = $city . ($country ? (', ' . $country) : '');

    $src = add_query_arg(
      [
        'key' => $key,
        'q'   => $q,
      ],
      'https://www.google.com/maps/embed/v1/place'
    );

    // min size 200x200 recommandé par Google :contentReference[oaicite:1]{index=1}
    $html  = '<div class="dcgm-map-wrap" style="margin:24px 0;">';
    $html .= '<div class="dcgm-map-title" style="font-weight:700;margin:0 0 10px;">';
    $html .= esc_html('Carte — ' . $city);
    $html .= '</div>';
    $html .= '<iframe loading="lazy" width="100%" height="' . esc_attr($height) . '" style="border:0" allowfullscreen ';
    $html .= 'referrerpolicy="no-referrer-when-downgrade" ';
    $html .= 'src="' . esc_url($src) . '"></iframe>';
    $html .= '</div>';

    return $html;
  }

  /* -------------------------
     Auto append to content
  ------------------------- */
  public static function append_map_to_content($content) {
    if (!is_singular('post') || !in_the_loop() || !is_main_query()) return $content;

    $opt = self::get_settings();
    if (empty($opt['auto_append'])) return $content;

    $post_id = get_the_ID();
    $city = get_post_meta($post_id, self::META_CITY, true);

    // si meta vide (ancien post), on tente à la volée
    if (!$city) {
      [$city2, $norm2] = self::detect_city_in_title(get_the_title($post_id));
      if ($city2) {
        $city = $city2;
        update_post_meta($post_id, self::META_CITY, $city2);
        update_post_meta($post_id, self::META_NORM, $norm2);
      }
    }

    if (!$city) return $content;

    $map = self::build_map_html($city);
    if ($map === '') return $content;

    return $content . $map;
  }

  /* -------------------------
     Shortcode [city_map]
  ------------------------- */
  public static function shortcode_city_map($atts) {
    if (!is_singular('post')) return '';

    $post_id = get_the_ID();
    $city = get_post_meta($post_id, self::META_CITY, true);

    if (!$city) {
      [$city2, $norm2] = self::detect_city_in_title(get_the_title($post_id));
      if ($city2) {
        $city = $city2;
        update_post_meta($post_id, self::META_CITY, $city2);
        update_post_meta($post_id, self::META_NORM, $norm2);
      }
    }

    return $city ? self::build_map_html($city) : '';
  }
}

DCGM_Dynamic_City_Google_Map::init();
