<?php
/**
 * Plugin Name: City Map Sidebar (auto from title + wp_villes) - Reliable
 * Description: Ajoute automatiquement une carte Google Maps en dernier widget de la sidebar sur les articles, en détectant la ville dans le titre via la table {prefix}villes.
 * Version: 1.1.0
 */

if (!defined('ABSPATH')) exit;

class City_Map_Sidebar_Auto {
  const TABLE_SUFFIX = 'villes'; // => {$wpdb->prefix}villes
  const META_CITY = '_cms_detected_city';
  const META_CITY_NORM = '_cms_detected_city_norm';

  public static function init() {
    add_filter('dynamic_sidebar_params', [__CLASS__, 'append_map_after_last_widget'], 9999);
  }

  public static function append_map_after_last_widget($params) {
    if (is_admin()) return $params;

    // Sidebar ciblée (celle que tu utilises dans le template)
    if (empty($params[0]['id']) || $params[0]['id'] !== 'sidebar-1') return $params;

    // Uniquement sur les articles
    if (!is_singular('post')) return $params;

    // Clé API obligatoire
    $api_key = self::get_api_key();
    if (!$api_key) return $params;

    // On récupère la liste des widgets dans cette sidebar
    $sidebars = wp_get_sidebars_widgets();
    $widgets = $sidebars['sidebar-1'] ?? [];
    if (empty($widgets) || !is_array($widgets)) return $params;

    // $params[0]['widget_id'] = id du widget en cours d'affichage
    $current_widget_id = $params[0]['widget_id'] ?? '';
    if ($current_widget_id === '') return $params;

    // On n'ajoute la carte qu'après le DERNIER widget
    $last_widget_id = end($widgets);
    if ($current_widget_id !== $last_widget_id) return $params;

    // Génère le HTML de la carte (peut être vide si aucune ville détectée)
    $map_html = self::render_map_widget();
    if (!$map_html) return $params;

    // On colle le widget "carte" juste après le dernier widget réel
    $params[0]['after_widget'] = ($params[0]['after_widget'] ?? '') . $map_html;

    return $params;
  }

  private static function render_map_widget() {
    $post_id = get_the_ID();
    if (!$post_id) return '';

    // Ville en cache ?
    $city = get_post_meta($post_id, self::META_CITY, true);

    if (!$city) {
      $title = get_the_title($post_id);
      [$city, $norm] = self::detect_city_from_title($title);
      if (!$city) return ''; // aucune ville détectée => rien à afficher

      update_post_meta($post_id, self::META_CITY, $city);
      update_post_meta($post_id, self::META_CITY_NORM, $norm);
    }

    $api_key = self::get_api_key();
    if (!$api_key) return '';

    // Indice pays pour éviter les ambiguïtés (mets "France" si besoin)
    $q = $city . ', Belgique';

    $src = add_query_arg(
      [
        'key' => $api_key,
        'q'   => $q,
      ],
      'https://www.google.com/maps/embed/v1/place'
    );

    // Widget HTML (class widget => prend ton style WP)
    $html  = '<section class="widget widget_city_map" style="margin-top:18px;">';
    $html .= '<h2 class="widget-title" style="padding:10px 12px;">' . esc_html('Carte — ' . $city) . '</h2>';
    $html .= '<div class="widget-content" style="padding:0 0 12px;">';
    $html .= '<iframe loading="lazy" width="100%" height="280" style="border:0; display:block;" allowfullscreen ';
    $html .= 'referrerpolicy="no-referrer-when-downgrade" src="' . esc_url($src) . '"></iframe>';
    $html .= '</div></section>';

    return $html;
  }

  private static function get_api_key() {
    // Méthode 1 : wp-config.php
    if (defined('GOOGLE_MAPS_EMBED_API_KEY') && GOOGLE_MAPS_EMBED_API_KEY) {
      return trim((string)GOOGLE_MAPS_EMBED_API_KEY);
    }

    // Méthode 2 : filtre (si tu préfères)
    $k = apply_filters('cms_google_maps_key', '');
    $k = is_string($k) ? trim($k) : '';
    return $k ?: '';
  }

  private static function detect_city_from_title($title) {
    global $wpdb;

    $title_norm = self::normalize($title);
    if ($title_norm === '') return [null, null];

    $table = $wpdb->prefix . self::TABLE_SUFFIX;

    // Cache 12h
    $cities = get_transient('cms_city_list_v2');
    if (!is_array($cities) || empty($cities)) {
      $rows = $wpdb->get_col("SELECT ville FROM {$table} WHERE ville IS NOT NULL AND ville <> ''");
      if (!is_array($rows)) $rows = [];

      $cities = [];
      foreach ($rows as $v) {
        $v = trim((string)$v);
        if ($v === '') continue;
        $norm = self::normalize($v);
        if ($norm === '') continue;
        // 1ere forme gardée
        if (!isset($cities[$norm])) $cities[$norm] = $v;
      }

      // Longueur décroissante => "Saint-Gilles" match avant "Gilles"
      uksort($cities, function($a, $b){
        return mb_strlen($b) <=> mb_strlen($a);
      });

      set_transient('cms_city_list_v2', $cities, 12 * HOUR_IN_SECONDS);
    }

    $hay = ' ' . $title_norm . ' ';

    foreach ($cities as $norm => $display) {
      $needle = ' ' . $norm . ' ';
      if (strpos($hay, $needle) !== false) {
        return [$display, $norm];
      }
    }

    return [null, null];
  }

  private static function normalize($s) {
    $s = trim((string)$s);
    if ($s === '') return '';

    $t = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
    if ($t !== false) $s = $t;

    $s = mb_strtolower($s);
    $s = preg_replace('~[^a-z0-9]+~', ' ', $s);
    $s = trim(preg_replace('~\s+~', ' ', $s));

    return $s;
  }
}

City_Map_Sidebar_Auto::init();
