<?php

// Active la prise en charge des widgets
function mon_theme_widgets_init() {
    register_sidebar( array(
        'name'          => 'Barre latérale principale',
        'id'            => 'sidebar-1',
        'description'   => 'Zone de widgets affichée dans la barre latérale.',
        'before_widget' => '<div id="%1$s" class="widget %2$s animate-on-scroll">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
}
add_action( 'widgets_init', 'mon_theme_widgets_init' );


function depannage_setup() {
  // Support titre automatique
  add_theme_support('title-tag');

  // Support miniatures
  add_theme_support('post-thumbnails');

  // Menus
  register_nav_menus([
    'primary' => __('Menu Principal'),
  ]);

  // Support HTML5
  add_theme_support('html5', ['search-form', 'comment-form', 'comment-list', 'gallery', 'caption']);
}

add_action('after_setup_theme', 'depannage_setup');

function depannage_scripts() {
  wp_enqueue_style('depannage-style', get_stylesheet_uri());
  wp_enqueue_script('depannage-script', get_template_directory_uri() . '/script.js', ['jquery'], null, true);
}
add_action('wp_enqueue_scripts', 'depannage_scripts');

// Menu responsive toggle
function depannage_menu_toggle_script() {
  ?>
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      const toggle = document.querySelector('.menu-toggle');
      const menu = document.getElementById('primary-menu');
      toggle.addEventListener('click', () => {
        const expanded = toggle.getAttribute('aria-expanded') === 'true' || false;
        toggle.setAttribute('aria-expanded', !expanded);
        menu.classList.toggle('active');
      });
    });
  </script>
  <?php
}


// Ajoute un champ image dans le formulaire d'ajout de catégorie
add_action('category_add_form_fields', function() {
    ?>
    <div class="form-field term-group">
        <label for="category-image"><?php _e('Image de la catégorie', 'textdomain'); ?></label>
        <input type="hidden" id="category-image" name="category-image" class="custom_media_url">
        <div id="category-image-wrapper"></div>
        <p>
            <input type="button" class="button button-secondary custom_media_button" id="custom_media_button" value="<?php _e('Ajouter une image', 'textdomain'); ?>" />
            <input type="button" class="button button-secondary custom_media_remove" value="<?php _e('Supprimer l\'image', 'textdomain'); ?>" />
        </p>
    </div>
    <?php
});

// Ajoute un champ image dans le formulaire d’édition de catégorie
add_action('category_edit_form_fields', function($term) {
    $image_id = get_term_meta($term->term_id, 'category-image-id', true);
    ?>
    <tr class="form-field term-group-wrap">
        <th scope="row"><label for="category-image"><?php _e('Image de la catégorie', 'textdomain'); ?></label></th>
        <td>
            <input type="hidden" id="category-image" name="category-image" value="<?php echo esc_attr($image_id); ?>">
            <div id="category-image-wrapper">
                <?php if ($image_id) echo wp_get_attachment_image($image_id, 'thumbnail'); ?>
            </div>
            <p>
                <input type="button" class="button button-secondary custom_media_button" id="custom_media_button" value="<?php _e('Ajouter une image', 'textdomain'); ?>" />
                <input type="button" class="button button-secondary custom_media_remove" value="<?php _e('Supprimer l\'image', 'textdomain'); ?>" />
            </p>
        </td>
    </tr>
    <?php
});

// Sauvegarde de l’image lors de la création/édition
add_action('created_category', function($term_id) {
    if (isset($_POST['category-image']) && '' !== $_POST['category-image']) {
        update_term_meta($term_id, 'category-image-id', absint($_POST['category-image']));
    }
});

add_action('edited_category', function($term_id) {
    if (isset($_POST['category-image']) && '' !== $_POST['category-image']) {
        update_term_meta($term_id, 'category-image-id', absint($_POST['category-image']));
    } else {
        delete_term_meta($term_id, 'category-image-id');
    }
});

// Script JS pour le média uploader
add_action('admin_footer', function() {
    if (!isset($_GET['taxonomy']) || $_GET['taxonomy'] != 'category') return;
    ?>
    <script>
    jQuery(document).ready(function($){
        function mediaUpload(button_class) {
            var _custom_media = true,
                _orig_send_attachment = wp.media.editor.send.attachment;
            $('body').on('click', button_class, function(e) {
                var button_id = $(this).attr('id');
                var send_attachment_bkp = wp.media.editor.send.attachment;
                var button = $(this);
                _custom_media = true;
                wp.media.editor.send.attachment = function(props, attachment){
                    if (_custom_media) {
                        $('#category-image').val(attachment.id);
                        $('#category-image-wrapper').html('<img class="custom_media_image" src="'+attachment.sizes.thumbnail.url+'" style="max-width:150px;"/>');
                    } else {
                        return _orig_send_attachment.apply(this, [props, attachment]);
                    }
                }
                wp.media.editor.open(button);
                return false;
            });
        }
        mediaUpload('.custom_media_button');
        $('body').on('click', '.custom_media_remove', function(){
            $('#category-image').val('');
            $('#category-image-wrapper').html('');
        });
    });
    </script>
    <?php
});

add_action('wp_footer', 'depannage_menu_toggle_script');




/**
 * Recherche par ville (liste déroulante) + résultats
 * Affiche les articles dont le TITRE contient la ville sélectionnée
 *
 * Utilisation dans un template :
 *   <?php rv_render_ville_search(); ?>
 */





function rv_get_villes() {
    return [
        'Aalst',
        'Aarschot',
        'Aartselaar',
        'Acoz',
        'Aiseau',
        'Aiseau-Presles',
        'Alleur',
        'Alsemberg',
        'Amay',
        'Amel-Amblève',
        'Andenne',
        'Anderlecht',
        'Anderlues',
        'Angleur',
        'Angre',
        'Angreau',
        'Anhée',
        'Annevoie-Rouillon',
        'Ans',
        'Anthisnes',
        'Antoing',
        'Arlon',
        'Assesse',
        'Ath',
        'Auderghem',
        'Awans',
        'Aywaille',
        'Baarle-Hertog',
        'Bastogne',
        'Beaumont',
        'Beauraing',
        'Beersel',
        'Beloeil',
        'Berchem-Sainte-Agathe',
        'Bernissart',
        'Bertogne',
        'Beyne-Heusay',
        'Bierges',
        'Binche',
        'Blégny',
        'Braine-l’Alleud',
        'Braine-le-Château',
        'Braine-le-Comte',
        'Braives',
        'Bruges',
        'Bruxelles',
        'Burdinne',
        'Celles',
        'Charleroi',
        'Châtelet',
        'Chaudfontaine',
        'Chièvres',
        'Chimay',
        'Ciney',
        'Colfontaine',
        'Comines-Warneton',
        'Courcelles',
        'Court-Saint-Étienne',
        'Couvin',
        'Daverdisse',
        'Dinant',
        'Dison',
        'Doische',
        'Donceel',
        'Dour',
        'Éghezée',
        'Enghien',
        'Esneux',
        'Etalle',
        'Evere',
        'Faimes',
        'Farciennes',
        'Fernelmont',
        'Ferrières',
        'Fexhe-le-Haut-Clocher',
        'Flémalle',
        'Fléron',
        'Florennes',
        'Florenville',
        'Fontaine-l’Évêque',
        'Fosses-la-Ville',
        'Frameries',
        'Frasnes-lez-Anvaing',
        'Froidchapelle',
        'Gedinne',
        'Geer',
        'Gembloux',
        'Genappe',
        'Gerpinnes',
        'Gesves',
        'Gouvy',
        'Grâce-Hollogne',
        'Grez-Doiceau',
        'Habay',
        'Hamoir',
        'Hamois',
        'Hannut',
        'Hastière',
        'Havelange',
        'Hélécine',
        'Herbeumont',
        'Héron',
        'Herstal',
        'Herve',
        'Honnelles',
        'Houffalize',
        'Houyet',
        'Incourt',
        'Ittre',
        'Jalhay',
        'Jodoigne',
        'Jurbise',
        'La Bruyère',
        'La Hulpe',
        'La Louvière',
        'Lasne',
        'Le Roeulx',
        'Léglise',
        'Lens',
        'Les Bons Villers',
        'Lessines',
        'Leuze-en-Hainaut',
        'Libin',
        'Libramont-Chevigny',
        'Liège',
        'Limbourg',
        'Lincent',
        'Lobbes',
        'Lontzen',
        'Manage',
        'Manhay',
        'Marche-en-Famenne',
        'Marchin',
        'Martelange',
        'Meix-devant-Virton',
        'Mettet',
        'Modave',
        'Momignies',
        'Mons',
        'Mont-de-l’Enclus',
        'Montigny-le-Tilleul',
        'Morlanwelz',
        'Mouscron',
        'Namur',
        'Nandrin',
        'Nassogne',
        'Neufchâteau',
        'Nivelles',
        'Ohey',
        'Olne',
        'Onhaye',
        'Oreye',
        'Ottignies-Louvain-la-Neuve',
        'Ouffet',
        'Paliseul',
        'Pecq',
        'Pepinster',
        'Perwez',
        'Philippeville',
        'Plombières',
        'Pont-à-Celles',
        'Profondeville',
        'Quévy',
        'Quaregnon',
        'Ramillies',
        'Rebecq',
        'Remicourt',
        'Rendeux',
        'Rixensart',
        'Rochefort',
        'Saint-Ghislain',
        'Saint-Hubert',
        'Saint-Nicolas',
        'Sambreville',
        'Schaerbeek',
        'Seneffe',
        'Seraing',
        'Soignies',
        'Somme-Leuze',
        'Sombreffe',
        'Spa',
        'Stavelot',
        'Stoumont',
        'Tellin',
        'Tenneville',
        'Theux',
        'Thimister-Clermont',
        'Thuin',
        'Tinlot',
        'Tintigny',
        'Trois-Ponts',
        'Trooz',
        'Tubize',
        'Vaux-sur-Sûre',
        'Verlaine',
        'Verviers',
        'Vielsalm',
        'Villers-la-Ville',
        'Virton',
        'Visé',
        'Walhain',
        'Wanze',
        'Waremme',
        'Waterloo',
        'Wavre',
        'Wellin',
        'Yvoir',
    ];
}





/** Nettoyage basique */
function rv_clean_ville($ville) {
    $ville = wp_strip_all_tags($ville);
    return trim($ville);
}

/**
 * Rend le bloc complet (formulaire + résultats)
 * @param array $args Options (facultatif)
 */
function rv_render_ville_search($args = []) {
    $defaults = [
        'posts_per_page' => 20,
        'post_type'      => 'post',
        'button_label'   => 'Rechercher',
        'label'          => 'Choisir une ville :',
        'placeholder'    => '- Ville -',
        'title_results'  => 'Nous intervenons bien à %s ! Appelez-nous au <a href="tel:0495991088">0495 99 10 88</a> ',
        'no_results'     => 'Aucun article trouvé dont le titre contient cette ville.',
        'invalid_city'   => 'Ville non valide.',
        'container_class'=> 'rv-wrap',
    ];
    $args = wp_parse_args($args, $defaults);

    $villes = rv_get_villes();

    // Ville sélectionnée via URL ?ville=...
    $selected = isset($_GET['ville']) ? rv_clean_ville(wp_unslash($_GET['ville'])) : '';

    // URL de la page courante
    $action_url = (function_exists('get_permalink') && get_queried_object_id())
        ? get_permalink(get_queried_object_id())
        : home_url(add_query_arg([]));

    echo '<div class="' . esc_attr($args['container_class']) . '">';

    // --- Formulaire
    ?>
    <form class="rv-form" method="get" action="<?php echo esc_url($action_url); ?>">
        <label for="rv-ville"><?php echo esc_html($args['label']); ?></label>

        <select name="ville" id="rv-ville" required>
            <option value=""><?php echo esc_html($args['placeholder']); ?></option>
            <?php foreach ($villes as $ville): ?>
                <option value="<?php echo esc_attr($ville); ?>" <?php selected($selected, $ville); ?>>
                    <?php echo esc_html($ville); ?>
                </option>
            <?php endforeach; ?>
        </select>

        <button class="rv-button" type="submit"><?php echo esc_html($args['button_label']); ?></button>
    </form>
    <?php

    // Si aucune ville choisie, on s’arrête ici
    if (empty($selected)) {
        echo '</div>';
        return;
    }

    // Optionnel mais conseillé : refuser une ville hors liste
    if (!in_array($selected, $villes, true)) {
        echo '<div class="rv-results"><p>' . esc_html($args['invalid_city']) . '</p></div>';
        echo '</div>';
        return;
    }

    // --- Résultats (recherche UNIQUEMENT dans le titre)
    $callback = function($where) use ($selected) {
        global $wpdb;
        $like = '%' . $wpdb->esc_like($selected) . '%';
        $where .= $wpdb->prepare(" AND {$wpdb->posts}.post_title LIKE %s ", $like);
        return $where;
    };
    add_filter('posts_where', $callback);

    $q = new WP_Query([
        'post_type'           => $args['post_type'],
        'post_status'         => 'publish',
        'posts_per_page'      => (int) $args['posts_per_page'],
        'ignore_sticky_posts' => true,
        'orderby'             => 'date',
        'order'               => 'DESC',
        's'                   => '', // on évite la recherche dans le contenu
        'no_found_rows'       => true,
    ]);

    remove_filter('posts_where', $callback);

    echo '<div class="rv-results">';
echo '<h3>' . wp_kses_post(sprintf($args['title_results'], esc_html($selected))) . '</h3>';

    if ($q->have_posts()) {
        echo '<ul>';
        while ($q->have_posts()) {
            $q->the_post();
            echo '<li><a href="' . esc_url(get_permalink()) . '">' . esc_html(get_the_title()) . '</a></li>';
        }
        echo '</ul>';
    } else {
        echo '<p>' . esc_html($args['no_results']) . '</p>';
    }

    echo '</div>';

    wp_reset_postdata();

    echo '</div>';
}

/**
 * CSS : injecté automatiquement sur le front (option 1).
 * Si tu préfères le mettre dans style.css, tu peux supprimer cette fonction.
 */
function rv_enqueue_ville_search_css() {
    if (is_admin()) return;

    $css = <<<CSS
/* ===== Recherche Ville (rv) ===== */
.rv-form{
  --rv-radius: 14px;
  --rv-border: rgba(20, 20, 20, .12);
  --rv-shadow: 0 10px 30px rgba(0,0,0,.06);

  display: grid;
  grid-template-columns: 1fr;
  gap: 10px;

  padding: 16px;
  border: 1px solid var(--rv-border);
  border-radius: var(--rv-radius);
  background: rgba(255,255,255,.75);
  box-shadow: var(--rv-shadow);
  backdrop-filter: blur(8px);
  -webkit-backdrop-filter: blur(8px);
}

@media (min-width: 720px){
  .rv-form{
    grid-template-columns: auto 1fr auto;
    align-items: center;
    gap: 12px;
    padding: 18px;
  }
}

.rv-form label{
  font-weight: 600;
  font-size: 0.95rem;
  letter-spacing: .2px;
  opacity: .9;
}

.rv-form select{
  width: 100%;
  height: 44px;
  padding: 0 14px;
  border-radius: 12px;
  border: 1px solid var(--rv-border);
  background: #fff;
  font-size: 1rem;
  outline: none;
  transition: transform .15s ease, box-shadow .15s ease, border-color .15s ease;
  box-shadow: 0 1px 0 rgba(0,0,0,.02);
}

.rv-form select:hover{
  box-shadow: 0 8px 18px rgba(0,0,0,.06);
  transform: translateY(-1px);
}

.rv-form select:focus{
  border-color: rgba(0,0,0,.28);
  box-shadow: 0 0 0 4px rgba(0,0,0,.08);
  transform: translateY(-1px);
}

.rv-button{
  height: 44px;
  padding: 0 16px;
  border-radius: 12px;
  border: 1px solid rgba(0,0,0,.12);
  background: #111;
  color: #fff;
  font-weight: 650;
  font-size: .98rem;
  letter-spacing: .2px;
  cursor: pointer;
  transition: transform .15s ease, box-shadow .15s ease, opacity .15s ease;
  box-shadow: 0 12px 26px rgba(0,0,0,.16);
}

.rv-button:hover{
  transform: translateY(-1px);
  box-shadow: 0 16px 36px rgba(0,0,0,.20);
}

.rv-button:active{
  transform: translateY(0px);
  opacity: .92;
}

.rv-button:focus-visible{
  outline: none;
  box-shadow: 0 0 0 4px rgba(0,0,0,.12), 0 16px 36px rgba(0,0,0,.20);
}

.rv-results{
  margin-top: 18px;
  padding: 16px;
  border: 1px solid rgba(20,20,20,.10);
  border-radius: 14px;
  background: #fff;
  box-shadow: 0 10px 30px rgba(0,0,0,.06);
}

.rv-results h3{
  margin: 0 0 10px 0;
  font-size: 1.1rem;
}

.rv-results ul{
  list-style: none;
  padding: 0;
  margin: 10px 0 0 0;
  display: grid;
  gap: 8px;
}

.rv-results li a{
  display: block;
  padding: 12px 14px;
  border-radius: 12px;
  border: 1px solid rgba(0,0,0,.08);
  text-decoration: none;
  color: inherit;
  transition: transform .15s ease, box-shadow .15s ease;
}

.rv-results li a:hover{
  transform: translateY(-1px);
  box-shadow: 0 10px 22px rgba(0,0,0,.08);
}
CSS;

    // On attache le CSS à une feuille existante (le style principal du thème)
    $handle = 'rv-inline-style';
    wp_register_style($handle, false);
    wp_enqueue_style($handle);
    wp_add_inline_style($handle, $css);
}
add_action('wp_enqueue_scripts', 'rv_enqueue_ville_search_css');

/**
 * Affiche les résultats de recherche ville juste après le H1
 * (pages et articles uniquement)
 */
add_filter('the_content', function ($content) {

    // uniquement sur le front
    if (is_admin()) {
        return $content;
    }

    // uniquement pages + articles
    if (!is_page() && !is_single()) {
        return $content;
    }

    // seulement si une ville est sélectionnée
    if (empty($_GET['ville'])) {
        return $content;
    }

    // on génère le HTML du formulaire + résultats
    ob_start();
    rv_render_ville_search();
    $search_html = ob_get_clean();

    // on injecte APRÈS le premier H1
    if (preg_match('/(<h1[^>]*>.*?<\/h1>)/is', $content, $matches)) {
        $content = preg_replace(
            '/(<h1[^>]*>.*?<\/h1>)/is',
            '$1' . $search_html,
            $content,
            1
        );
    } else {
        // fallback si aucun H1
        $content = $search_html . $content;
    }

    return $content;
});



