<?php
/**
 * Plugin Name: City Map Sidebar (auto from title + wp_villes)
 * Description: Ajoute automatiquement une carte Google Maps en dernier widget de la sidebar sur les articles, en détectant la ville dans le titre via la table wp_villes.
 * Version: 1.0.0
 */

if (!defined('ABSPATH')) exit;

class City_Map_Sidebar_Auto {
  // ⚠️ Mets ici ta table si ton préfixe WP n'est pas wp_
  // Laisse vide pour utiliser $wpdb->prefix automatiquement.
  const TABLE_SUFFIX = 'villes'; // => {$wpdb->prefix}villes

  // Post meta cache
  const META_CITY = '_cms_detected_city';
  const META_CITY_NORM = '_cms_detected_city_norm';

  public static function init() {
    add_filter('dynamic_sidebar_params', [__CLASS__, 'inject_map_widget_last'], 9999);
  }

  /** Injecte un "widget" HTML en dernier dans sidebar-1 */
  public static function inject_map_widget_last($params) {
    if (is_admin()) return $params;

    // On cible la sidebar principale (ajuste si besoin)
    if (empty($params[0]['id']) || $params[0]['id'] !== 'sidebar-1') return $params;

    // Seulement sur les articles
    if (!is_singular('post')) return $params;

    // Clé API dans wp-config.php ou via filtre
    $api_key = self::get_api_key();
    if (!$api_key) return $params;

    // On ajoute une action qui s'exécute juste après le dernier widget rendu
    // (WP appelle dynamic_sidebar_params pour chaque widget, on ne veut l'ajouter qu'une fois)
    static $hooked = false;
    if (!$hooked) {
      $hooked = true;
      add_action('dynamic_sidebar_after', function($sidebar_id){
        if ($sidebar_id !== 'sidebar-1') return;
        echo City_Map_Sidebar_Auto::render_map_widget();
      }, 9999, 1);
    }

    return $params;
  }

  /** Rendu du "dernier widget" */
  public static function render_map_widget() {
    $post_id = get_the_ID();
    if (!$post_id) return '';

    $city = get_post_meta($post_id, self::META_CITY, true);

    if (!$city) {
      // détecte à la volée + met en cache
      $title = get_the_title($post_id);
      [$city, $norm] = self::detect_city_from_title($title);

      if (!$city) return ''; // pas de ville -> pas de widget

      update_post_meta($post_id, self::META_CITY, $city);
      update_post_meta($post_id, self::META_CITY_NORM, $norm);
    }

    $api_key = self::get_api_key();
    if (!$api_key) return '';

    // Ajoute "Belgique" pour éviter ambiguïtés (change si besoin)
    $q = $city . ', Belgique';

    $src = add_query_arg(
      [
        'key' => $api_key,
        'q'   => $q,
      ],
      'https://www.google.com/maps/embed/v1/place'
    );

    // Widget HTML
    ob_start(); ?>
      <section class="widget widget_city_map" style="margin-top:18px;">
        <h2 class="widget-title" style="padding:10px 12px;">Carte — <?php echo esc_html($city); ?></h2>
        <div class="widget-content" style="padding:0 0 12px;">
          <iframe
            loading="lazy"
            width="100%"
            height="280"
            style="border:0; display:block;"
            allowfullscreen
            referrerpolicy="no-referrer-when-downgrade"
            src="<?php echo esc_url($src); ?>">
          </iframe>
        </div>
      </section>
    <?php
    return ob_get_clean();
  }

  /** Récupère la clé API (méthode la plus simple et rapide) */
  private static function get_api_key() {
    // 1) Recommandé: définir dans wp-config.php :
    // define('GOOGLE_MAP_
